#!/usr/bin/env node
/**
 * Convert a .4q file into .4v and .set components.
 * If file starts with "ICE!", uncompress first using unice68.exe.
 * Usage: node convert4q.js input.4q
 */

const fs = require('fs');
const path = require('path');
const { spawnSync } = require('child_process');

function abort(message) {
  console.error('Error:', message);
  process.exit(1);
}

if (process.argv.length < 3) {
  abort('Usage: node convert4q.js <inputfile.4q>');
}

let inputPath = process.argv[2];
if (!fs.existsSync(inputPath)) {
  abort(`File not found: ${inputPath}`);
}

/**
 * STEP 1  Check if the file starts with "ICE!".
 * If so, uncompress using unice68.exe and use the new file instead.
 */
function maybeUncompress(filePath) {
  const fd = fs.openSync(filePath, 'r');
  const header = Buffer.alloc(4);
  fs.readSync(fd, header, 0, 4, 0);
  fs.closeSync(fd);

  if (header.toString('ascii') === 'ICE!') {
    console.log('ICE! header detected  uncompressing with unice68.exe...');

    const exe = 'unice68.exe';
    const dir = path.dirname(filePath);
    const baseName = path.basename(filePath, path.extname(filePath));
    const decompressedPath = path.join(dir, `${baseName}.uniced`);

    const result = spawnSync(exe, ['-d', filePath, decompressedPath], {
      stdio: 'inherit',
    });

    if (result.error) {
      abort(`Failed to execute ${exe}: ${result.error.message}`);
    }
    if (result.status !== 0) {
      abort(`${exe} returned error code ${result.status}`);
    }
    if (!fs.existsSync(decompressedPath)) {
      abort('Uncompression failed  output file not found.');
    }

    console.log(`Using uncompressed file: ${decompressedPath}`);
    return decompressedPath;
  }

  return filePath;
}

// Uncompress if needed
inputPath = maybeUncompress(inputPath);

/**
 * STEP 2  Proceed with the standard .4q format handling.
 */

let buffer;
try {
  buffer = fs.readFileSync(inputPath);
} catch (err) {
  abort(`Failed to read file: ${err.message}`);
}

// 1) Verify header is "QUARTET"
const header = buffer.subarray(0, 7).toString('ascii');
if (header !== 'QUARTET') {
  abort('Invalid file header: expected "QUARTET"');
}

// 2) Read big-endian length fields from offsets 8 and 12
if (buffer.length < 16) {
  abort('File too short to contain required header and lengths.');
}
const len4v = buffer.readUInt32BE(8);
const lenSet = buffer.readUInt32BE(12);

// 3) Extract data blocks with bounds validation
const offset4v = 20;
const offsetSet = offset4v + len4v;
const totalSizeNeeded = offsetSet + lenSet;

if (buffer.length < totalSizeNeeded) {
  abort(
    `File too short for expected data. Expected ${totalSizeNeeded} bytes, got ${buffer.length}.`
  );
}

const data4v = buffer.subarray(offset4v, offset4v + len4v);
const dataSet = buffer.subarray(offsetSet, offsetSet + lenSet);

// 4) Write output files
const baseName = path.basename(inputPath, path.extname(inputPath));
const dirName = path.dirname(inputPath);
const out4vPath = path.join(dirName, `${baseName}.4v`);
const outSetPath = path.join(dirName, `${baseName}.set`);

try {
  fs.writeFileSync(out4vPath, data4v);
  fs.writeFileSync(outSetPath, dataSet);
  console.log(`Created: ${out4vPath}`);
  console.log(`Created: ${outSetPath}`);
} catch (err) {
  abort(`Error writing output files: ${err.message}`);
}

console.log('Conversion successful.');